/*===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 */
#include <vdb/extern.h>

#include <vdb/xform.h>
#include <vdb/database.h>
#include <vdb/table.h>
#include <vdb/cursor.h>
#include <vdb/vdb-priv.h>
#include <insdc/insdc.h>
#include <klib/data-buffer.h>
#include <klib/rc.h>
#include <sysalloc.h>

#include <bitstr.h>

#include <stdlib.h>
#include <string.h>
#include <assert.h>


typedef struct RefTableSubSelect RefTableSubSelect;
struct RefTableSubSelect
{
    const VCursor *curs;
    uint32_t out_idx;
    uint32_t circular_idx;
    uint32_t name_idx;
    uint32_t name_range_idx;
};

static
void CC RefTableSubSelectWhack ( void *obj )
{
    RefTableSubSelect * self = obj;
    if ( self != NULL )
    {
        VCursorRelease ( self -> curs );
        free ( self );
    }
}

static
rc_t RefTable_READ_SubSelectMake ( RefTableSubSelect **objp, const VTable *tbl )
{
    rc_t rc;

    /* create the object */
    RefTableSubSelect *obj = malloc ( sizeof * obj );
    if ( obj == NULL ) {
        rc = RC ( rcXF, rcFunction, rcConstructing, rcMemory, rcExhausted );
    } else {
        /* get at the parent database */
        const VDatabase *db;
        rc = VTableOpenParentRead ( tbl, & db );
        if ( rc == 0 )
        {
            /* open the reference table */
            const VTable *reftbl;
            rc = VDatabaseOpenTableRead ( db, & reftbl, "REFERENCE" );
            VDatabaseRelease ( db );
            if ( rc == 0 )
            {
                /* create a cursor */
                rc = VTableCreateCachedCursorRead(reftbl, &obj->curs, 4*1024*1024);
                VTableRelease ( reftbl );
                if( rc == 0 ) {
                    /* add columns to cursor */
                    if( (rc = VCursorAddColumn(obj->curs, &obj->out_idx,  "(INSDC:4na:bin) READ")) == 0 &&
                        (rc = VCursorAddColumn(obj->curs, &obj->name_idx, "(utf8)NAME")) == 0 &&
                        (rc = VCursorAddColumn(obj->curs, &obj->circular_idx, "CIRCULAR")) == 0 &&
                        (rc = VCursorAddColumn(obj->curs, &obj->name_range_idx, "NAME_RANGE")) == 0 ) {
                        rc = VCursorOpen(obj->curs);
                    }
                    if( rc == 0 ) {
                        *objp = obj;
                        return 0;
                    }
                    VCursorRelease ( obj -> curs );
                }
            }
        }
        free ( obj );
    }
    return rc;
}

static
rc_t RefTable_PRESERVE_QUAL_SubSelectMake ( RefTableSubSelect **objp, const VTable *tbl )
{
    rc_t rc;
    
    /* create the object */
    RefTableSubSelect *obj = malloc ( sizeof * obj );
    if ( obj == NULL ) {
        rc = RC ( rcXF, rcFunction, rcConstructing, rcMemory, rcExhausted );
    } else {
        /* get at the parent database */
        const VDatabase *db;
        rc = VTableOpenParentRead ( tbl, & db );
        if ( rc == 0 )
        {
            /* open the reference table */
            const VTable *reftbl;
            rc = VDatabaseOpenTableRead ( db, & reftbl, "REFERENCE" );
            VDatabaseRelease ( db );
            if ( rc == 0 )
            {
                /* create a cursor */
                rc = VTableCreateCachedCursorRead(reftbl, &obj->curs, 4*1024*1024);
                VTableRelease ( reftbl );
                if( rc == 0 ) {
                    /* add columns to cursor */
                    if( (rc = VCursorAddColumn(obj->curs, &obj->out_idx,  "(bool)PRESERVE_QUAL")) == 0 &&
                       (rc = VCursorAddColumn(obj->curs, &obj->name_idx, "(utf8)NAME")) == 0 &&
                       (rc = VCursorAddColumn(obj->curs, &obj->circular_idx, "CIRCULAR")) == 0 &&
                       (rc = VCursorAddColumn(obj->curs, &obj->name_range_idx, "NAME_RANGE")) == 0 ) {
                        rc = VCursorOpen(obj->curs);
                    }
                    if( rc == 0 ) {
                        *objp = obj;
                        return 0;
                    }
                    VCursorRelease ( obj -> curs );
                }
            }
        }
        free ( obj );
    }
    return rc;
}

static void my_copy(void *Dst, bitsz_t doff, void const *Src, bitsz_t soff, bitsz_t sz)
{
    if ((doff & 7) == 0 && (soff & 7) == 0 && (sz & 7) == 0) {
        uint8_t *const dst = (uint8_t *)Dst;
        uint8_t const *const src = (uint8_t const *)Src;
        
        memcpy(&dst[doff >> 3], &src[soff >> 3], sz >> 3);
    }
    else
        bitcpy(Dst, doff, Src, soff, sz);
}

static
rc_t CC reftbl_sub_select ( void *data, const VXformInfo *info,
    int64_t row_id, VRowResult *rslt, uint32_t argc, const VRowData argv[] )
{
    rc_t rc;
    const RefTableSubSelect * self = (const void*)data;
    uint8_t *dst;
    uint32_t num_read, row_len, ref_len;
    INSDC_coord_zero offset;
    int64_t ref_row_id;
    char* name = NULL;
    uint32_t name_len = 0;
    const bool* circular;

    /* get start and length of reference segment */
    const int64_t* ref_id = argv[0].u.data.base;
    const INSDC_coord_zero* ref_start = argv[1].u.data.base;
    const INSDC_coord_len* ref_lenv  = argv[2].u.data.base;

    assert(argv[0].u.data.elem_bits == sizeof(*ref_id) * 8 );
    assert(argv[1].u.data.elem_bits == sizeof(*ref_start) * 8 );
    assert(argv[2].u.data.elem_bits == sizeof(*ref_lenv) * 8 );

    ref_lenv += argv[2].u.data.first_elem;
    ref_len = ref_lenv[0];
    /* get the memory for output row */
    
    rslt->data->elem_bits = rslt->elem_bits;
    if((rc = KDataBufferResize ( rslt->data, ref_len )) == 0 ) {
        dst = rslt->data->base;
        /* starting row and offset */
        ref_id += argv[0].u.data.first_elem;
        ref_start += argv[1].u.data.first_elem;
        ref_row_id = ref_id[0];
        offset = ref_start[0];
    }
    /* read the sequence */
    for( num_read = 0; rc == 0 && num_read < ref_len; offset = 0, num_read += row_len ) {
        VCursorCloseRow(self->curs);
        if( (rc = VCursorSetRowId(self->curs, ref_row_id)) == 0 &&
            (rc = VCursorOpenRow(self->curs)) == 0 ) {
            void const* output;
            const char* name_next;
            uint32_t name_len_next;
            uint32_t bits;
            uint32_t boff;
            
            if( (rc = VCursorCellData(self->curs, self->out_idx, &bits, &output, &boff, &row_len)) == 0 &&
                (rc = VCursorCellData(self->curs, self->name_idx, NULL, (void const **)&name_next, NULL, &name_len_next)) == 0 ) {
                if( name != NULL && (name_len != name_len_next || memcmp(name, name_next, name_len) != 0) ) {
                    /* next name detected */
                    if( *circular ) {
                        /* use i_name to find 1st rowid */
                        if( (rc = VCursorParamsSet((const struct VCursorParams *)(self->curs),
                                                   "QUERY_SEQ_NAME", "%.*s", name_len, name)) == 0 ) {
                            struct {
                                int64_t start_id;
                                uint64_t id_count;
                            } *out;
                            if( (rc = VCursorCellData(self->curs, self->name_range_idx, NULL, (const void**)&out, NULL, NULL)) == 0 ) {
                                row_len = 0;
                                ref_row_id = out->start_id;
                            }
                        }
                        continue;
                    } else {
                        rc = RC(rcXF, rcFunction, rcSelecting, rcData, rcCorrupt);
                    }
                } else if( name == NULL ) {
                    uint32_t l;
                    if( (rc = VCursorCellData(self->curs, self->circular_idx, NULL,
                                              (const void**)&circular, NULL, &l)) == 0 ) {
                        assert(l == sizeof(*circular));
                        name = malloc(name_len_next);
                        if( name == NULL ) {
                            rc = RC(rcXF, rcFunction, rcSelecting, rcMemory, rcExhausted);
                        } else {
                            memcpy(name, name_next, name_len_next);
                            name_len = name_len_next;
                        }
                    }
                }
                if( rc == 0 ) {
                    /* row_len MUST be > offset */
                    if( row_len <= offset ) {
                        rc = RC(rcXF, rcFunction, rcSelecting, rcData, rcCorrupt);
                    } else {
                        row_len -= offset;
                        if(ref_len < row_len + num_read) {
                            row_len = ref_len - num_read;
                        }
                        /* copy data */
                        my_copy(dst, num_read * bits, output, offset * bits + boff, row_len * bits);

                        ref_row_id++;
                    }
                }
            }
        }
    }
    /* detect incomplete read */
    if( rc == 0 && num_read < ref_len ) {
        rc = RC(rcXF, rcFunction, rcSelecting, rcTransfer, rcIncomplete);
    }
    free(name);
    rslt->elem_count = num_read;
    return rc;
}

VTRANSFACT_IMPL ( ALIGN_ref_sub_select, 1, 0, 0 ) ( const void *self, const VXfactInfo *info,
    VFuncDesc *rslt, const VFactoryParams *cp, const VFunctionParams *dp )
{
    RefTableSubSelect *fself;
    rc_t rc = RefTable_READ_SubSelectMake ( & fself, info -> tbl );
    if ( rc == 0 )
    {
        rslt -> self = fself;
        rslt -> whack = RefTableSubSelectWhack;
        rslt -> u . ndf = reftbl_sub_select;
        rslt -> variant = vftNonDetRow;
    }

    return rc;
}

VTRANSFACT_IMPL ( NCBI_align_ref_sub_select_preserve_qual, 1, 0, 0 ) ( const void *self, const VXfactInfo *info,
                                                   VFuncDesc *rslt, const VFactoryParams *cp, const VFunctionParams *dp )
{
    RefTableSubSelect *fself;
    rc_t rc = RefTable_PRESERVE_QUAL_SubSelectMake ( & fself, info -> tbl );
    if ( rc == 0 )
    {
        rslt -> self = fself;
        rslt -> whack = RefTableSubSelectWhack;
        rslt -> u . ndf = reftbl_sub_select;
        rslt -> variant = vftNonDetRow;
    }
    
    return rc;
}
