#include "speaker.h"
#include <QDebug>

Speaker::Speaker() :
    fs(0.0),
    vas(0.0),
    re(0.0),
    qts(0.0),
    sd(0.0),
    xmax(0.0)
{
}

Speaker::Speaker(const Speaker &copy)
{
    this->fs = copy.getFs();
    this->vas = copy.getVas();
    this->re = copy.getRe();
    this->qts = copy.getQts();
    this->sd = copy.getSd();
    this->xmax = copy.getXmax();

    this->vendor = copy.getVendor();
    this->model = copy.getModel();
}

Speaker::~Speaker()
{
    /* nothing to do, just for QMetatype */
}

bool Speaker::operator!=(const Speaker& r) const
{
    return this->fs != r.getFs() ||
            this->vas != r.getVas() ||
            this->re != r.getRe() ||
            this->qts != r.getQts() ||
            this->sd != r.getSd() ||
            this->xmax != r.getXmax();
}

bool Speaker::operator==(const Speaker& r) const
{
    return !(*this != r);
}

bool Speaker::isValid() const
{
    return !vendor.isNull() && !model.isNull();
}

void Speaker::setVendor(const QString &vnd)
{
    vendor = vnd;
}

void Speaker::setModel(const QString &mdl)
{
    model = mdl;
}

void Speaker::setFs(double val)
{
    fs = val;
}

void Speaker::setVas(double val)
{
    vas = val;
}

void Speaker::setRe(double val)
{
    re = val;
}

void Speaker::setQts(double val)
{
    qts = val;
}

void Speaker::setSd(double val)
{
    sd = val;
}

void Speaker::setXmax(double val)
{
    xmax = val;
}

QString Speaker::getVendor() const
{
    return vendor;
}

QString Speaker::getModel() const
{
    return model;
}

double Speaker::getFs() const
{
    return fs;
}

double Speaker::getVas() const
{
    return vas;
}

double Speaker::getRe() const
{
    return re;
}

double Speaker::getQts() const
{
    return qts;
}

double Speaker::getSd() const
{
    return sd;
}

double Speaker::getXmax() const
{
    return xmax;
}

QDomElement Speaker::toDomElement(QDomDocument &doc) const
{
    QDomElement e = doc.createElement("speaker");

    e.setAttribute("vendor", vendor);
    e.setAttribute("model", model);

    QLocale c(QLocale::C);
    e.setAttribute("fs", c.toString(fs));
    e.setAttribute("vas", c.toString(vas));
    e.setAttribute("re", c.toString(re));
    e.setAttribute("qts", c.toString(qts));
    e.setAttribute("sd", c.toString(sd));
    e.setAttribute("xmax", c.toString(xmax));

    return e;
}

void Speaker::fromDomElement(const QDomElement &el)
{
    vendor = el.attribute("vendor");
    model = el.attribute("model");

    QLocale c(QLocale::C);
    fs = c.toDouble(el.attribute("fs"));
    vas = c.toDouble(el.attribute("vas"));
    re = c.toDouble(el.attribute("re"));
    qts = c.toDouble(el.attribute("qts"));
    sd = c.toDouble(el.attribute("sd"));
    xmax = c.toDouble(el.attribute("xmax"));
}

void Speaker::render(QPainter *painter, const QRectF &area)
{
    if (!painter)
        return;

    painter->drawRoundRect(area.toRect(), 5, 5);

    QString params[6];

    params[0] = QString::fromUtf8("Fs: %1 Hz").arg(getFs());
    params[1] = QString::fromUtf8("Qts: %1").arg(getQts());
    params[2] = QString::fromUtf8("Vas: %1 L").arg(getVas());
    params[3] = QString::fromUtf8("Sd: %1 m²").arg(getSd());
    params[4] = QString::fromUtf8("Xmax: %1 mm").arg(getXmax());
    params[5] = QString::fromUtf8("Re: %1 Ohm").arg(getRe());

    qreal tab = area.left();

    QString text = getVendor() + " " + getModel();
    QRectF where(area.left(), area.top(), area.width(), area.height() / 2);
    QTextOption option(Qt::AlignCenter);
    painter->drawText(where, text, option);

    for (int i = 0; i < 6; i++) {
        where.setRect(tab, area.top() + area.height() / 2, area.width() / 6, area.height() / 2);
        text = params[i];
        option.setAlignment(Qt::AlignVCenter|Qt::AlignLeft);
        painter->drawText(where, text, option);
        tab += area.width() / 6;
    }
}
